/* Copyright (c) 2016, NVIDIA CORPORATION.  All rights reserved.
 *
 * NVIDIA CORPORATION and its licensors retain all intellectual property
 * and proprietary rights in and to this software, related documentation
 * and any modifications thereto.  Any use, reproduction, disclosure or
 * distribution of this software and related documentation without an express
 * license agreement from NVIDIA CORPORATION is strictly prohibited.
 */

'use strict'

var api = require('./NvSDKAPINode.node');

var _logger;

module.exports = function (app, io, logger) {
    if (app === undefined || io === undefined || logger == undefined) {
        throw 'You need to provide express app, socket io and logging';
    }

    //! Allows global access to logging module
    _logger = logger;

    //! Helper function converts the returned error string into a JSON object
    //! @param errorType      String type of error
    //! @param errorString    String of error message
    //! @return               JSON object of error data
    function BuildErrorObject(errorType, errorString) {
        var returnCode;
        var returnText;
        var returnMessage;
        var parts = errorString.split("::");
        for (var i = 0; i < parts.length; i++) {
            var index = parts[i].search("Text:")
            if (index != -1) {
                returnText = parts[i].substring(index + 5);
            }
            index = parts[i].search("Code:")
            if (index != -1) {
                returnCode = Number(parts[i].substring(index + 5));
            }
            index = parts[i].search("Message:")
            if (index != -1) {
                returnMessage = parts[i].substring(index + 8);
            }
        }
        var errorResult = { type: errorType, code: returnCode, codeText: returnText, message: returnMessage };
        _logger.info(errorResult);
        return errorResult;
    }

    //! Helper function that receives body of POST request and calls callback for that data.
    //! @param req      Request object provided by Express.
    //! @param callback Callback that is triggered on succesfully downloaded data.
    function getPostDataAndDo(req, callback) {
        var content = ''

        function onData(data) {
            content += data;
            _logger.info('Post Data: ' + content);
        }

        function onEnd() {
            callback(content);
        }

        req.on('data', onData);
        req.on('end', onEnd);
    }

    //! Helper function that receives JSON body of POST request and calls callback for the parsed data.
    //! @param req      Request object provided by Express.
    //! @param res      Response object provided by Express.
    //! @param callback Callback that is triggered on succesfully parsed data.
    function getJSONDataAndDo(req, res, callback) {
        function parseAndCallback(content) {
            var parsed = {};
            try {
                parsed = JSON.parse(content);
            }
            catch (e) {
                res.writeHead(400, { 'Content-Type': 'application/json' });
                var errorResult = BuildErrorObject(e.name, e.message);
                var errorString = JSON.stringify(errorResult);
                _logger.error(errorString);
                res.end(errorString);
                return;
            }
            callback(parsed);
        }
        getPostDataAndDo(req, parseAndCallback);
    }

    app.get('/SDK/v.1.0/Highlights/GroupActive', function (req, res) {
        function doReply(err, data) {
            if (err) {
                replyWithError(res, err);
            }
            else {
                res.writeHead(200, { 'Content-Type': 'application/json' });
                res.end(JSON.stringify(data));
            }
        }
        try {
            api.GetGroupActive(doReply);
        }
        catch (err) {
            replyWithError(res, err);
        }
    });

    app.get('/SDK/v.1.0/Enabled', function (req, res) {
        function doReply(err, data) {
            if (err) {
                replyWithError(res, err);
            }
            else {
                res.writeHead(200, { 'Content-Type': 'application/json' });
                res.end(JSON.stringify(data));
            }
        }
        try {
            api.GetEnabled(doReply);
        }
        catch (err) {
            replyWithError(res, err);
        }
    });

    // Callback Handling
    function EmitNotification(name, data) {
        setImmediate(function () { io.emit(name, data); });
    }

    function CreateHighlightsNotificationCallback(data) {
        logger.info('Highlights Callback', JSON.stringify(data));
        EmitNotification('/SDK/v.1.0/Highlights', data);
    }
    api.SetOscHighlightsCallback(CreateHighlightsNotificationCallback);

    //! Returns the version of this module
    //! This must be the last declaration in the file. Anything below it will be masked.
    return {
        version: function () {
            return api.GetVersion();
        },
        Cleanup: function () {
            api.Cleanup();
        }
    };
};
